<?php

namespace App\Http\Controllers\Admin;

use App\Exceptions\MembershipException;
use App\Http\Controllers\Controller;
use App\Models\Lesson;
use App\Services\MembershipService;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Validation\Rules\File;
use Illuminate\Support\Facades\Storage;

class MembershipController extends Controller
{
    private MembershipService $mem;

    public function __construct()
    {
        $this->mem = new MembershipService();
    }

    public function addCourse(Request $request): RedirectResponse
{
    if (!$request->hasFile('image') && !$request->hasFile('image_url')) {
        return redirect()->back()->with('message', 'Please choose a course image');
    }

    \Log::info('Starting addCourse', [
        'has_image' => $request->hasFile('image'),
        'has_image_url' => $request->hasFile('image_url')
    ]);

    $path = null;
    $file = null;

    if ($request->hasFile('image')) {
        $file = $request->file('image');
    } else if ($request->hasFile('image_url')) {
        $file = $request->file('image_url');
    }

    if ($file) {
        $this->validate($request, [
            'title' => ['required'],
            'desc' => ['required'],
        ]);

        \Log::info('Processing file', [
            'original_name' => $file->getClientOriginalName()
        ]);

        $filename = time() . '_' . $file->getClientOriginalName();
        $file->move(storage_path('app/public/courses'), $filename);
        $path = 'storage/courses/' . $filename;
        
        \Log::info('After move', ['path' => $path]);
    }

    try {
        \Log::info('Before addCourse', ['path' => $path]);
        $message = $this->mem->addCourse([
            'title' => $request->title,
            'price' => $request->amount,
            'image_url' => $path,
            'is_paid' => !empty($request->amount) ? true : false,
            'category_id' => $request->category,
            'description' => $request->desc,
        ]);

        return to_route('admin.membership.courses', ['page' => 1])->with('success', $message);
    } catch (MembershipException $ex) {
        \Log::error('Error in addCourse', ['error' => $ex->getMessage()]);
        return back()->with('message', $ex->getMessage());
    }
}

    public function editCourse(Request $request): RedirectResponse
    {
        if (empty($request->image_url) && !$request->hasFile('image')) {
            return redirect()->back()->with('message', 'Please choose a course image');
        }

        $path = $request->image_url;

        if ($request->hasFile('image')) {
            $this->validate($request, [
                'image' => [
                    'nullable',
                    File::image()
                        ->min('1kb')
                        ->max('10mb'),
                ],
                'title' => ['required'],
                'desc' => ['required'],
            ]);

            $file = $request->file('image');
            $path = $file->store('courses', 'public');
            $path = 'storage/' . $path;
        }

        try {
            $message = $this->mem->updateCourse([
                'title' => $request->title,
                'price' => $request->amount,
                'image_url' => $path,
                'is_paid' => (bool)!empty($request->amount),
                'category_id' => $request->category,
                'description' => $request->desc,
            ], $request->course_id);

            return to_route('admin.membership.courses', ['page' => 1])->with('success', $message);
        } catch (MembershipException $ex) {
            return back()->with('message', $ex->getMessage());
        }
    }


    public function deleteCourse(int $courseId): RedirectResponse
    {
        try {
            $message = $this->mem->deleteCourse($courseId);
            return to_route('admin.membership.courses', ['page' => 1])->with('success', $message);
        } catch (MembershipException $ex) {
            return back()->with('message', $ex->getMessage());
        }
    }

    public function addLesson(Request $request): RedirectResponse
    {
        $request->validate([
        'category' => 'required|string|max:255',
        'title' => 'required|string|max:255',
        'length' => 'required|string|max:50',
        'videolink' => 'required|url',
        'preview' => 'required|boolean',
        'course_id' => 'nullable|integer',
        'desc' => 'required|string',
        'image' => 'nullable|file|image|max:10240',
        'image_url' => 'nullable|string',
    ]);

        $path = $request->image_url;

        if ($request->hasFile('image')) {
            $this->validate($request, [
                'image' => [
                    'nullable',
                    File::image()
                        ->min('1kb')
                        ->max('10mb'),
                ],
            ]);
            $file = $request->file('image');
            $path = $file->store('lessons');
        }

        try {
            $message = $this->mem->addLesson([
                'category' => $request->category,
                'title' => $request->title,
                'length' => $request->length,
                'video_link' => $request->videolink,
                'preview' => $request->preview,
                'course_id' => $request->course_id,
                'description' => $request->desc,
                'thumbnail' => $path,
            ]);
            return back()->with('success', $message);
        } catch (MembershipException $ex) {
            return back()->with('message', $ex->getMessage());
        }
    }
    
    public function updateLesson(Request $request): RedirectResponse
{
    $request->validate([
        'title' => 'required|string|max:255',
        'length' => 'required|string',
        'videolink' => 'required|url',
        'descc' => 'required|string',
        'preview' => 'required|boolean',
        'lesson_id' => 'required|integer',
        'course_id' => 'required|integer',
        'image' => 'nullable|file|image|max:10240',
        'image_url' => 'nullable|string',
    ]);

    try {
        $lesson = Lesson::findOrFail($request->lesson_id);

        $lesson->title = $request->title;
        $lesson->length = $request->length;
        $lesson->video_link = $request->videolink;
        $lesson->description = $request->descc;
        $lesson->preview = $request->preview;

       $path = $request->image_url;

        if ($request->hasFile('image')) {
            $this->validate($request, [
                'image' => [
                    'nullable',
                    File::image()
                        ->min('1kb')
                        ->max('10mb'),
                ],
            ]);
            $file = $request->file('image');
            $path = $file->store('lessons');
        }

        $lesson->save();

        return redirect()->route('admin.membership.courseLessons', ['id' => $request->course_id, 'page' => 1])
            ->with('success', 'Lesson updated successfully.');
    } catch (\Exception $e) {
        return back()->with('error', 'Failed to update lesson: ' . $e->getMessage());
    }
}


    public function deleteLesson(int $lessonId, int $courseId = null): RedirectResponse
    {
        try {
            $message = $this->mem->deleteLesson($lessonId);
            return back()->with('success', $message);
        } catch (MembershipException $ex) {
            return back()->with('message', $ex->getMessage());
        }
    }
}
