<?php

namespace App\Livewire\Admin\InvestmentPlan;

use App\Models\Plan;
use App\Models\User;
use App\Models\UserPlan;
use App\Models\Transaction;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Cache;
use Jantinnerezo\LivewireAlert\LivewireAlert;
use Livewire\Attributes\Computed;
use Livewire\Component;

class AddProfit extends Component
{
    use LivewireAlert;

    public $amount;
    public $amount_type = 'Percent';
    public $useAmount = 'plan_increment_amount';
    public $addToBalance = 1;
    public $plan;
    public $userPlan;
    public $type = 'all';

    public function render()
    {
        return view('livewire.admin.investment-plan.add-profit');
    }

    #[Computed]
    public function plans(): Collection
    {
        return Plan::latest()->get();
    }

    #[Computed]
    public function usersPlans(): Collection
    {
        return UserPlan::with('user:id,name', 'plan:id,name')->where('status', 'active')->latest()->get();
    }

    public function save(): void
    {
        $this->authorize('manually add profit');

        $this->validate([
            'plan' => ['exclude_if:type,single', 'required', 'exists:plans,id'],
            'userPlan' => ['exclude_if:type,all', 'required', 'exists:user_plans,id'],
            'amount' => ['nullable', 'numeric'],
        ]);

        if ($this->type === 'all') {
            try {
                $usersPlans = UserPlan::with('user', 'plan')->where('plan_id', $this->plan)->where('status', 'active')->get();
                
                $usersPlans->each(function (UserPlan $userPlan) {
                    $this->creditProfit($userPlan);
                });
                
                $this->flash(message: 'Profit added successfully.', redirect: route('admin.usersPlans'));
            } catch (\Exception $e) {
                $this->alert(type: 'error', message: $e->getMessage());
            }
        }

        if ($this->type === 'single') {
            try {
                $userPlan = UserPlan::with('user', 'plan')->find($this->userPlan);
                $this->creditProfit($userPlan);
                $this->flash(message: 'Profit added successfully.', redirect: route('admin.usersPlans'));
            } catch (\Exception $e) {
                $this->alert(type: 'error', message: $e->getMessage());
            }
        }
    }

    private function creditProfit(UserPlan $userPlan): void
    {
        try {
            $incrementValues = explode(',', $userPlan->plan->increment_amount);
            $increment_amount = $incrementValues[array_rand($incrementValues)];
            
            // Calculate profit amount based on settings
            $amount = $this->calculateProfitAmount($userPlan->amount, $increment_amount);
            
            // Save profit to database
            $userPlan->rois()->create([
                'user_id' => $userPlan->user_id,
                'amount' => $amount,
            ]);
            
            $user = User::select('id', 'account_bal', 'roi')->find($userPlan->user_id);
            
            if ((bool) $this->addToBalance) {
                $user->increment('account_bal', $amount);
                $user->increment('roi', $amount);
                
                // Record the transaction
                Transaction::create([
                    'user_id' => $user->id,
                    'narration' => "ROI profit from {$userPlan->plan->name} investment plan",
                    'amount' => $amount,
                    'type' => 'Credit',
                ]);
            } else {
                $user->increment('roi', $amount);
            }
            
            $userPlan->increment('profit_earned', $amount);
            
            // Clear relevant caches if needed
            Cache::forget('total_roi_profits');
            Cache::forget('chart_investments');
            
        } catch (\Throwable $th) {
            throw new \Exception($th->getMessage());
        }
    }
    
    private function calculateProfitAmount(float $capital, float $incrementAmount): float
    {
        $amount = 0;
        
        if ($this->useAmount === 'plan_increment_amount') {
            $amount = $capital * ($incrementAmount / 100);
        } else {
            if ($this->amount_type === 'Percent') {
                $amount = $capital * (floatval($this->amount) / 100);
            } else {
                $amount = floatval($this->amount);
            }
        }
        
        return round($amount, 2);
    }
}